'use client';

import { useState, useEffect, Suspense } from 'react';
import { TemplateConfig } from '@/types';
import Link from 'next/link';
import { useSearchParams } from 'next/navigation';
import { ArrowLeft, Save, Smartphone, Monitor, Check, Edit, Eye, MessageCircle, Globe, Palette } from 'lucide-react';

// Import Templates
import LandscapingTemplate from '@/components/templates/LandscapingTemplate';
import MovingTemplate from '@/components/templates/MovingTemplate';
import UsedFurnitureTemplate from '@/components/templates/UsedFurnitureTemplate';
import ContractingTemplate from '@/components/templates/ContractingTemplate';
import RecruitmentTemplate from '@/components/templates/RecruitmentTemplate';
import DecorTemplate from '@/components/templates/DecorTemplate';
import PestControlTemplate from '@/components/templates/PestControlTemplate';

function BuilderContent({ template }: { template: TemplateConfig }) {
    const searchParams = useSearchParams();
    const projectId = searchParams.get('id');

    const [data, setData] = useState<Record<string, string>>(() => {
        const initial: Record<string, string> = {};
        template.fields.forEach(f => {
            if (f.defaultValue) initial[f.id] = f.defaultValue;
        });
        if (!initial.primaryColor) initial.primaryColor = '#16a34a';
        return initial;
    });

    const [viewMode, setViewMode] = useState<'desktop' | 'mobile'>('desktop');
    const [isPublishing, setIsPublishing] = useState(false);
    const [publishedUrl, setPublishedUrl] = useState<string | null>(null);
    const [activeTab, setActiveTab] = useState<'editor' | 'preview'>('editor');
    const [currentProjectId, setCurrentProjectId] = useState<string | null>(projectId);

    // Fetch existing project data if editing
    useEffect(() => {
        if (projectId) {
            fetch('/api/projects')
                .then(res => res.json())
                .then((projects: any[]) => {
                    const project = projects.find(p => p.id === projectId);
                    if (project) {
                        setData(project.data);
                        setCurrentProjectId(project.id);
                        // If it's already published, we might want to show the link?
                        // For now, let's just allow re-publishing.
                    }
                });
        }
    }, [projectId]);

    const handleChange = (id: string, value: string) => {
        setData(prev => ({ ...prev, [id]: value }));
    };

    const handleImageUpload = (id: string, e: React.ChangeEvent<HTMLInputElement>) => {
        const file = e.target.files?.[0];
        if (file) {
            const reader = new FileReader();
            reader.onloadend = () => {
                setData(prev => ({ ...prev, [id]: reader.result as string }));
            };
            reader.readAsDataURL(file);
        }
    };

    const handlePublish = async () => {
        setIsPublishing(true);
        try {
            const payload: any = {
                templateId: template.id,
                name: data.companyName || template.name,
                data: data
            };

            // If we have a currentProjectId, send it to update instead of create
            if (currentProjectId) {
                payload.id = currentProjectId;
            }

            const res = await fetch('/api/projects', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(payload)
            });

            if (res.ok) {
                const project = await res.json();
                const url = `${window.location.origin}/p/${project.slug}`;
                setPublishedUrl(url);
                setCurrentProjectId(project.id); // Update ID in case it was new
            } else {
                const err = await res.json();
                console.error('Server error:', err);
                alert('فشل النشر: ' + (err.error || 'خطأ غير معروف'));
            }
        } catch (error) {
            console.error('Failed to publish', error);
            alert('حدث خطأ أثناء النشر');
        } finally {
            setIsPublishing(false);
        }
    };

    const handleExport = () => {
        const htmlContent = document.getElementById('preview-container')?.innerHTML;
        if (htmlContent) {
            const fullHtml = `
<!DOCTYPE html>
<html dir="rtl" lang="ar">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>${data.seoTitle || data.companyName || 'Landing Page'}</title>
    <meta name="description" content="${data.seoDescription || ''}">
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Cairo:wght@400;700&display=swap');
        body { font-family: 'Cairo', sans-serif; }
    </style>
</head>
<body>
    ${htmlContent}
    ${data.whatsapp ? `
    <a href="https://wa.me/${data.whatsapp}" target="_blank" class="fixed bottom-6 right-6 bg-green-500 text-white p-4 rounded-full shadow-lg hover:bg-green-600 transition-colors z-50 flex items-center justify-center">
        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 11.5a8.38 8.38 0 0 1-.9 3.8 8.5 8.5 0 0 1-7.6 4.7 8.38 8.38 0 0 1-3.8-.9L3 21l1.9-5.7a8.38 8.38 0 0 1-.9-3.8 8.5 8.5 0 0 1 4.7-7.6 8.38 8.38 0 0 1 3.8-.9h.5a8.48 8.48 0 0 1 8 8v.5z"></path></svg>
    </a>
    ` : ''}
</body>
</html>`;

            const blob = new Blob([fullHtml], { type: 'text/html' });
            const url = URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = `${template.id}-landing-page.html`;
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(url);
        }
    };

    return (
        <div className="flex flex-col md:flex-row h-screen bg-gray-100 overflow-hidden" dir="rtl">

            {/* Mobile Tabs */}
            <div className="md:hidden flex bg-white border-b border-gray-200">
                <button
                    onClick={() => setActiveTab('editor')}
                    className={`flex-1 py-3 text-sm font-bold flex items-center justify-center gap-2 ${activeTab === 'editor' ? 'text-blue-600 border-b-2 border-blue-600' : 'text-gray-500'}`}
                >
                    <Edit className="w-4 h-4" />
                    المحرر
                </button>
                <button
                    onClick={() => setActiveTab('preview')}
                    className={`flex-1 py-3 text-sm font-bold flex items-center justify-center gap-2 ${activeTab === 'preview' ? 'text-blue-600 border-b-2 border-blue-600' : 'text-gray-500'}`}
                >
                    <Eye className="w-4 h-4" />
                    المعاينة
                </button>
            </div>

            {/* Sidebar Editor */}
            <div className={`
        w-full md:w-96 bg-white border-l border-gray-200 flex flex-col shadow-lg z-10
        ${activeTab === 'editor' ? 'flex' : 'hidden md:flex'}
        h-full md:h-auto
      `}>
                <div className="p-4 border-b border-gray-200 flex items-center justify-between bg-gray-50">
                    <Link href="/" className="text-gray-500 hover:text-gray-700 flex items-center gap-2 text-sm font-medium">
                        <ArrowLeft className="w-4 h-4" />
                        العودة
                    </Link>
                    <h2 className="font-bold text-gray-800">محرر الصفحة</h2>
                </div>

                <div className="flex-1 overflow-y-auto p-6 space-y-8">
                    {publishedUrl ? (
                        <div className="bg-green-50 border border-green-200 rounded-lg p-4 text-center">
                            <div className="w-12 h-12 bg-green-100 text-green-600 rounded-full flex items-center justify-center mx-auto mb-3">
                                <Check className="w-6 h-6" />
                            </div>
                            <h3 className="font-bold text-green-800 mb-2">تم النشر بنجاح!</h3>
                            <p className="text-sm text-green-700 mb-4">صفحتك جاهزة الآن على الرابط:</p>
                            <div className="bg-white p-2 rounded border border-green-200 text-xs break-all mb-3 select-all">
                                {publishedUrl}
                            </div>
                            <a href={publishedUrl} target="_blank" className="block w-full bg-green-600 text-white py-2 rounded text-sm font-bold hover:bg-green-700">
                                زيارة الصفحة
                            </a>
                            <button onClick={() => setPublishedUrl(null)} className="mt-2 text-xs text-gray-500 underline">
                                متابعة التعديل
                            </button>
                        </div>
                    ) : (
                        <>
                            {/* Main Settings Section */}
                            <div className="space-y-4">
                                <h3 className="font-bold text-gray-900 flex items-center gap-2 pb-2 border-b border-gray-100">
                                    <Palette className="w-4 h-4 text-blue-600" />
                                    المظهر والإعدادات
                                </h3>

                                {/* Primary Color */}
                                <div className="space-y-2">
                                    <label className="block text-sm font-medium text-gray-700">اللون الرئيسي</label>
                                    <div className="flex gap-2">
                                        <input
                                            type="color"
                                            value={data.primaryColor || '#16a34a'}
                                            onChange={(e) => handleChange('primaryColor', e.target.value)}
                                            className="h-10 w-20 rounded cursor-pointer"
                                        />
                                        <input
                                            type="text"
                                            value={data.primaryColor || '#16a34a'}
                                            onChange={(e) => handleChange('primaryColor', e.target.value)}
                                            className="flex-1 p-2 border border-gray-300 rounded-md text-sm uppercase"
                                        />
                                    </div>
                                </div>

                                {/* WhatsApp */}
                                <div className="space-y-2">
                                    <label className="block text-sm font-medium text-gray-700">رقم واتساب (للزرد العائم)</label>
                                    <div className="relative">
                                        <MessageCircle className="absolute right-3 top-2.5 w-4 h-4 text-gray-400" />
                                        <input
                                            type="text"
                                            value={data.whatsapp || ''}
                                            onChange={(e) => handleChange('whatsapp', e.target.value)}
                                            className="w-full p-2 pr-10 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 outline-none text-sm"
                                            placeholder="966500000000"
                                        />
                                    </div>
                                </div>
                            </div>

                            {/* SEO Section */}
                            <div className="space-y-4">
                                <h3 className="font-bold text-gray-900 flex items-center gap-2 pb-2 border-b border-gray-100">
                                    <Globe className="w-4 h-4 text-blue-600" />
                                    تحسين محركات البحث (SEO)
                                </h3>

                                <div className="space-y-2">
                                    <label className="block text-sm font-medium text-gray-700">عنوان الصفحة (Google Title)</label>
                                    <input
                                        type="text"
                                        value={data.seoTitle || ''}
                                        onChange={(e) => handleChange('seoTitle', e.target.value)}
                                        className="w-full p-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 outline-none text-sm"
                                        placeholder="أفضل شركة تنظيف بالرياض - خصم 20%"
                                    />
                                </div>

                                <div className="space-y-2">
                                    <label className="block text-sm font-medium text-gray-700">وصف الصفحة (Meta Description)</label>
                                    <textarea
                                        value={data.seoDescription || ''}
                                        onChange={(e) => handleChange('seoDescription', e.target.value)}
                                        className="w-full p-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 outline-none text-sm"
                                        rows={3}
                                        placeholder="وصف مختصر يظهر في نتائج البحث..."
                                    />
                                </div>
                            </div>

                            {/* Template Fields */}
                            <div className="space-y-4">
                                <h3 className="font-bold text-gray-900 flex items-center gap-2 pb-2 border-b border-gray-100">
                                    <Edit className="w-4 h-4 text-blue-600" />
                                    محتوى الصفحة
                                </h3>
                                {template.fields.map((field) => (
                                    <div key={field.id} className="space-y-2">
                                        <label className="block text-sm font-medium text-gray-700">
                                            {field.label} {field.required && <span className="text-red-500">*</span>}
                                        </label>
                                        {field.type === 'textarea' ? (
                                            <textarea
                                                value={data[field.id] || ''}
                                                onChange={(e) => handleChange(field.id, e.target.value)}
                                                className="w-full p-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500 outline-none transition-all text-sm"
                                                rows={3}
                                                placeholder={field.placeholder}
                                            />
                                        ) : field.type === 'image' ? (
                                            <div className="border-2 border-dashed border-gray-300 rounded-md p-4 text-center hover:bg-gray-50 transition-colors cursor-pointer relative group">
                                                <input
                                                    type="file"
                                                    accept="image/*"
                                                    onChange={(e) => handleImageUpload(field.id, e)}
                                                    className="absolute inset-0 w-full h-full opacity-0 cursor-pointer z-10"
                                                />
                                                {data[field.id] ? (
                                                    <div className="relative h-32 w-full">
                                                        <img src={data[field.id]} alt="Preview" className="h-full w-full object-contain" />
                                                        <button
                                                            onClick={(e) => {
                                                                e.preventDefault();
                                                                handleChange(field.id, '');
                                                            }}
                                                            className="absolute top-0 right-0 bg-red-500 text-white p-1 rounded-full text-xs z-20 opacity-0 group-hover:opacity-100 transition-opacity"
                                                        >
                                                            X
                                                        </button>
                                                    </div>
                                                ) : (
                                                    <div className="flex flex-col items-center justify-center h-20 text-gray-400">
                                                        <span className="text-xs">اضغط لرفع صورة</span>
                                                    </div>
                                                )}
                                            </div>
                                        ) : (
                                            <input
                                                type={field.type === 'tel' ? 'tel' : 'text'}
                                                value={data[field.id] || ''}
                                                onChange={(e) => handleChange(field.id, e.target.value)}
                                                className="w-full p-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500 outline-none transition-all text-sm"
                                                placeholder={field.placeholder}
                                            />
                                        )}
                                    </div>
                                ))}
                            </div>
                        </>
                    )}
                </div>

                <div className="p-4 border-t border-gray-200 bg-gray-50 space-y-3">
                    <button
                        onClick={handlePublish}
                        disabled={isPublishing || !!publishedUrl}
                        className="w-full bg-blue-600 hover:bg-blue-700 disabled:bg-blue-400 text-white py-2 px-4 rounded-md flex items-center justify-center gap-2 font-medium transition-colors"
                    >
                        {isPublishing ? 'جاري النشر...' : 'حفظ ونشر (رابط تلقائي)'}
                    </button>

                    <button
                        onClick={handleExport}
                        className="w-full bg-white border border-gray-300 hover:bg-gray-50 text-gray-700 py-2 px-4 rounded-md flex items-center justify-center gap-2 font-medium transition-colors"
                    >
                        <Save className="w-4 h-4" />
                        تصدير الصفحة (HTML)
                    </button>
                </div>
            </div>

            {/* Preview Area */}
            <div className={`
        flex-1 flex-col h-full relative bg-gray-100
        ${activeTab === 'preview' ? 'flex' : 'hidden md:flex'}
      `}>
                <div className="bg-white border-b border-gray-200 p-3 flex justify-center gap-4 shadow-sm z-10">
                    <button
                        onClick={() => setViewMode('desktop')}
                        className={`p-2 rounded transition-colors ${viewMode === 'desktop' ? 'bg-blue-100 text-blue-600' : 'hover:bg-gray-100 text-gray-600'}`}
                        title="عرض سطح المكتب"
                    >
                        <Monitor className="w-5 h-5" />
                    </button>
                    <button
                        onClick={() => setViewMode('mobile')}
                        className={`p-2 rounded transition-colors ${viewMode === 'mobile' ? 'bg-blue-100 text-blue-600' : 'hover:bg-gray-100 text-gray-600'}`}
                        title="عرض الجوال"
                    >
                        <Smartphone className="w-5 h-5" />
                    </button>
                </div>
                <div className="flex-1 overflow-y-auto p-4 md:p-8 flex justify-center items-start">
                    <div
                        className={`bg-white shadow-2xl transition-all duration-300 overflow-hidden ${viewMode === 'mobile' ? 'w-[375px] h-[667px] rounded-3xl border-8 border-gray-800 shrink-0' : 'w-full min-h-full rounded-lg'
                            }`}
                    >
                        <div id="preview-container" className="w-full h-full overflow-y-auto bg-white relative">
                            {template.id === 'landscaping' && <LandscapingTemplate data={data} />}
                            {template.id === 'moving' && <MovingTemplate data={data} />}
                            {template.id === 'used-furniture' && <UsedFurnitureTemplate data={data} />}
                            {template.id === 'contracting' && <ContractingTemplate data={data} />}
                            {template.id === 'recruitment' && <RecruitmentTemplate data={data} />}
                            {template.id === 'decor' && <DecorTemplate data={data} />}
                            {template.id === 'pest-control' && <PestControlTemplate data={data} />}

                            {/* Floating WhatsApp Button Preview */}
                            {data.whatsapp && (
                                <div className="absolute bottom-6 right-6 bg-green-500 text-white p-4 rounded-full shadow-lg z-50 flex items-center justify-center">
                                    <MessageCircle className="w-6 h-6" />
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </div>
    );
}

export default function BuilderClient({ template }: { template: TemplateConfig }) {
    return (
        <Suspense fallback={<div>Loading...</div>}>
            <BuilderContent template={template} />
        </Suspense>
    );
}
